/*
 * ReadyGenerator.c
 *
 * Created: 03.11.2011 20:53:18
 *  Author: Christoph Raab
-- This program is free software; you can redistribute it and/or modify it under the terms of the 
-- GNU General Public License as published by the Free Software Foundation; either version 3 of the 
-- License, or (at your option) any later version.
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; 
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
-- See the GNU General Public License for more details.
-- You should have received a copy of the GNU General Public License along with this program; 
-- if not, see <http://www.gnu.org/licenses/>. 

-- Dieses Programm ist freie Software. Sie knnen es unter den Bedingungen der GNU General Public License, 
-- wie von der Free Software Foundation verffentlicht, weitergeben und/oder modifizieren, entweder gem 
-- Version 3 der Lizenz oder (nach Ihrer Option) jeder spteren Version.
-- Die Verffentlichung dieses Programms erfolgt in der Hoffnung, da es Ihnen von Nutzen sein wird, aber 
-- OHNE IRGENDEINE GARANTIE, sogar ohne die implizite Garantie der MARKTREIFE oder der VERWENDBARKEIT FR 
-- EINEN BESTIMMTEN ZWECK. Details finden Sie in der GNU General Public License.
-- Sie sollten ein Exemplar der GNU General Public License zusammen mit diesem Programm erhalten haben. 
-- Falls nicht, siehe <http://www.gnu.org/licenses/>. 
 
 */ 

#include <avr/io.h>
#include <avr/interrupt.h>

#define UEBERLAUF 2000
#define MAX_VERZOEGERUNG 1000					// angenommene max. Motorhochlaufzeit, wenn SEL inaktiv ist
#define MAX_INDEXPAUSE_HD 440					// Bei einer HD Diskette dauert eine Umdrehung etwa 400ms
#define MAX_INDEXPAUSE_DD 220					// Bei einer DD Diskette dauert eine Umdrehung etwa 200ms

volatile unsigned int stopuhr,verzoegerung0,verzoegerung1;


/* Zuordnung der Pins:

READY_GEN	: PB1	- dieses Signal teilt dem CPLD mit, dass es READY aktivieren kann
MOTOR_ON0	: PA0	- dieses Signal ist das Motorsignal von Laufwerk 0 (nicht das Amiga Motoron)
MOTOR_ON1	: PA1	- dieses Signal ist das Motorsignal von Laufwerk 1 (nicht das Amiga Motoron)
SEL0		: PA2	- Selekt Signal 0
SEL1		: PA3	- Selekt Signal 1
INDEX		: PB2	- Index-Signal vom Laufwerk
HDDISKIN	: PA7	- HD oder DD
DISKCHANGE	: PB0	- Diskchange

*/

#define READY_GEN_PIN PB1
#define MOTOR_ON0_PIN PA0
#define MOTOR_ON1_PIN PA1
#define SEL0_PIN PA2
#define SEL1_PIN PA3
#define INDEX_PIN PB2
#define HDDISKIN_PIN PA7
#define DISKCHANGE_PIN PB0

/* Zustandsautomat:

0. Motor aus

1. Motor luft hoch, SEL aktiv

2. Motor luft hoch, SEL inaktiv

3. Motor luft auf Drehzahl, SEL aktiv

4. Motor luft auf Drehzahl, SEL inaktiv

*/

uint8_t zustand0, zustand1;					// Variablen fr Zustandsautomat

volatile uint8_t READY_GEN0, READY_GEN1;			// interne Variable fr den Zustand "READY" der beiden Laufwerke
volatile uint8_t MOTOR_ON0, MOTOR_ON1; 				// Abbild der Eingnge
volatile uint8_t SEL0, SEL1, DISKCHANGE, HDDISKIN;		// Abbild der Eingnge

volatile uint8_t motor_auf_drehzahl;

int main(void)
{
	// Initialisierung
  
	// Timer 0 konfigurieren
	TCCR0A = (1<<WGM01); 					// CTC Modus - d.h. Zhlerregister zhlt hoch bis der Wert von OCR0A erreicht ist. Dann wird ein Interrupt ausgelst
	TCCR0B |= (1<<CS01); 					// Prescaler 8 - die interne Clock (1MHz) wird durch 8 geteilt, d.h. das Zhlerregister wird mit 125kHz ausgelesen
	OCR0A = 125-1;						// Das Vergleichsregister wird auf 125 gesetzt, d.h. alle 1ms wird ein Interrupt ausgelst
	// Compare Interrupt erlauben
	TIMSK0 |= (1<<OCIE0A);					// Interruptbehandlung fr Timer-Interrupt wird aktiviert
 
	stopuhr = UEBERLAUF;					// zeitmesser auf Oberkante setzen
	
	verzoegerung0 = 0;
	verzoegerung1 = 0;
	
	zustand0 = 0;						// Automaten starten im Zustand 'Motor aus'
	zustand1 = 0;
	
	READY_GEN0 = 1; 					// Ready anfangs auf 1
	READY_GEN1 = 1;
	
	DDRA = 0x00;						// Port A ist komplett Eingang
	PORTA = 0x00;
	
	DDRB = 0x02;						// Port B, Pin1 ist Ausgang
	PORTB = 0x02;						// ReadyGen Ausgang wird auf 1 gesetzt.
	
	
	MCUCR = (MCUCR & 0xfc) | 1 << ISC01 | 0 << ISC00;	// INT0 (INDEX_PIN) wird bei fallender Flanke ausgelst
	GIMSK = 1 << INT0;					// Interruptbehandlung fr INT0-Interrupt wird aktiviert
	
	
	sei();							// Interrupts werden global aktiviert
	    
	while(1)
	{
		MOTOR_ON0 = (PINA & (1 << MOTOR_ON0_PIN));	// Pins abfragen
		MOTOR_ON1 = (PINA & (1 << MOTOR_ON1_PIN));
		SEL0 = (PINA & (1 << SEL0_PIN));
		SEL1 = (PINA & (1 << SEL1_PIN));
		
		DISKCHANGE = (PINB & (1 << DISKCHANGE_PIN));
		
		HDDISKIN = (PINA & (1 << HDDISKIN_PIN));
		
		switch(zustand0)
		{								// Automat fr Laufwerk 0
		  case 0:							// Zustand: Motor aus
			READY_GEN0 = 1;						// Ready aus
			if (!MOTOR_ON0)						// Wenn Motor gestartet wird...
			{
				if (!SEL0)					// und SEL0 aktiv ist
					{
						stopuhr = UEBERLAUF;		// erstes Index-Ereignis soll nicht zu Ready aktiv fhren
						motor_auf_drehzahl = 0;
						verzoegerung0 = 0;		// Verzgerungszeit (bei SEL =1) zurcksetzen
						zustand0 = 1;			// und ab in Zustand 1
					}					
				else						// und SEL0 inaktiv (wird in der Praxis vermutlich nicht auftreten)
					{
						stopuhr = UEBERLAUF;		// erstes Index-Ereignis soll nicht zu Ready aktiv fhren
						motor_auf_drehzahl = 0;
						verzoegerung0 = 0;		// Verzgerungszeit (bei SEL =1) zurcksetzen
						zustand0 = 2;			// und ab in Zustand 1
					}
			}							// Wenn Motor nicht gestartet wird bleiben im Zustand 0
			break;
		case 1:
			READY_GEN0 = 1;						// Ready aus
			if (!MOTOR_ON0)						// Wenn Motor luft....
			{
				if (!SEL0)					// und SEL0 aktiv ist
				{
					if (motor_auf_drehzahl)			// Wenn der Motor auf Drehzahl ist (dieser Wert wird in der Interruptroutine zu INT0 erzeugt)
						zustand0 = 3;			// dann ab in Zustand 3
				}
				else
				zustand0 = 2;					// und SEL0 inaktiv -> Zustand 2
			}
			else
				zustand0 = 0;					// Motor aus -> Zustand 0
			break;
		case 2:
			READY_GEN0 = 1;						// Ready aus
			if (!MOTOR_ON0)						// Wenn Motor luft....
			{
				if (SEL0)					// und SEL0 inaktiv ist
				{
					if (verzoegerung0 > MAX_VERZOEGERUNG)	// Wenn seit dem start MAX_VERZOEGERUNG vergangen ist
						zustand0 = 4;			// dann ab in Zustand 4
				}				
				else
				{						// SEL wurde aktiv
					stopuhr = UEBERLAUF;			// Stopuhr fr Index auf berlauf setzen
					motor_auf_drehzahl = 0;
					zustand0 = 1;				// und SEL0 aktiv -> Zustand 1
				}				
			}
			else
				zustand0 = 0;					// Motor aus -> Zustand 0
			break;
		case 3:
			if (!MOTOR_ON0)						// Wenn Motor luft....
			{
				READY_GEN0 = 0;					// Ready an - endlich !
				if (!SEL0)					// und SEL0 aktiv ist
				{
					if (!motor_auf_drehzahl)		// Wenn nicht mehr auf Drehzahl
						zustand0 = 1;			// zurck in Zustand 1
				}
				else
					zustand0 = 4;				// und SEL0 inaktiv -> Zustand 4
			}
			else
				zustand0 = 0;					// Motor aus -> Zustand 0
			break;
		case 4:
			if (!MOTOR_ON0)						// Wenn Motor luft....
			{
				READY_GEN0 = 0;					// Ready an - endlich !
				if (!SEL0)					// und SEL0 aktiv geworden ist
				{
					stopuhr = 0;				//  Stopuhr zurcksetzen
					motor_auf_drehzahl = 1;
					zustand0 = 3;				// und ab in Zustand 3
				}						// Wenn SEL inaktiv geblieben ist, bleiben wir in Zustand 4
			}
			else
				zustand0 = 0;					// Motor aus -> Zustand 0
			break;
		}		

		switch(zustand1)
		{								// Automat fr Laufwerk 1
		case 0:								// Motor aus
			READY_GEN1 = 1;						// Ready aus
			if (!MOTOR_ON1)						// Wenn Motor gestartet wird...
			{
				if (!SEL1)					// und SEL1 aktiv ist
					{
						stopuhr = UEBERLAUF;		// erstes Index-Ereignis soll nicht zu Ready aktiv fhren
						motor_auf_drehzahl = 0;
						verzoegerung1 = 0;		// Verzgerungszeit (bei SEL =1) zurcksetzen
						zustand1 = 1;			// und ab in Zustand 1
					}					
				else						// und SEL1 inaktiv (wird in der Praxis vermutlich nicht auftreten)
					{
						stopuhr = UEBERLAUF;		// erstes Index-Ereignis soll nicht zu Ready aktiv fhren
						motor_auf_drehzahl = 0;
						verzoegerung1 = 0;		// Verzgerungszeit (bei SEL =1) zurcksetzen
						zustand1 = 2;			// und ab in Zustand 1
					}
			}							// Wenn Motor nicht gestartet wird bleiben im Zustand 0
			break;
		case 1:
			READY_GEN1 = 1;						// Ready aus
			if (!MOTOR_ON1)						// Wenn Motor luft....
			{
				if (!SEL1)					// und SEL1 aktiv ist
				{
					if (motor_auf_drehzahl)			// Wenn der Motor auf Drehzahl ist (dieser Wert wird in der Interruptroutine zu INT0 erzeugt)
						zustand1 = 3;			// dann ab in Zustand 3
				}
				else
					zustand1 = 2;				// und SEL1 inaktiv -> Zustand 2
			}
			else
				zustand1 = 0;					// Motor aus -> Zustand 0
			break;
		case 2:
			READY_GEN1 = 1;						// Ready aus
			if (!MOTOR_ON1)						// Wenn Motor luft....
			{
				if (SEL1)					// und SEL1 inaktiv ist
				{
					if (verzoegerung1 > MAX_VERZOEGERUNG)	// Wenn seit dem start MAX_VERZOEGERUNG vergangen ist
						zustand1 = 4;			// dann ab in Zustand 4
				}
				else
				{
					stopuhr = UEBERLAUF;			// Stopuhr fr Index auf berlauf setzen
					motor_auf_drehzahl = 0;
					zustand1 = 1;				// und SEL1 aktiv -> Zustand 1
				}				
			}
			else
				zustand1 = 0;					// Motor aus -> Zustand 0
			break;
		case 3:
			if (!MOTOR_ON1)						// Wenn Motor luft....
			{
				READY_GEN1 = 0;						// Ready an - endlich !
				if (!SEL1)						// und SEL1 aktiv ist
				{
					if (!motor_auf_drehzahl)		// Wenn der Motor nicht mehr auf Drehzahl ist
					zustand1 = 1;				// zurck in Zustand 1
				}
				else
					zustand1 = 4;				// und SEL1 inaktiv -> Zustand 4
			}
			else
				zustand1 = 0;					// Motor aus -> Zustand 0
			break;
		case 4:
			if (!MOTOR_ON1)						// Wenn Motor luft....
			{
				READY_GEN1 = 0;					// Ready an - endlich !
				if (!SEL1)					// und SEL1 aktiv geworden ist
				{
					stopuhr = 0;				//  Stopuhr zurcksetzen
					motor_auf_drehzahl = 1;
					zustand1 = 3;				// und ab in Zustand 3
				}
			}
			else
				zustand1 = 0;					// Motor aus -> Zustand 0
			break;
		}

		if ((!SEL0 & !READY_GEN0)|(!SEL1 & !READY_GEN1))
			PORTB = 0 << READY_GEN_PIN;
		else
			PORTB = 1 << READY_GEN_PIN;
	}
}

/*
Der Compare Interrupt Handler 
wird aufgerufen, wenn 
TCNT0 = OCR0A = 125-1 
ist (125 Schritte), d.h. genau alle 1 ms
*/
ISR (TIM0_COMPA_vect)
{
	stopuhr++;								// Uhren werden hoch gezhlt
	if (stopuhr > UEBERLAUF) stopuhr=UEBERLAUF;				// und auf berlauf begrenzt
	verzoegerung0++;
	if (verzoegerung0 > UEBERLAUF) verzoegerung0=UEBERLAUF;
	verzoegerung1++;
	if (verzoegerung1 > UEBERLAUF) verzoegerung1=UEBERLAUF;
}

ISR (SIG_INTERRUPT0)								// Interrupt bei fallender Index-Flanke
{
	if (!HDDISKIN)
	{
		if (stopuhr < MAX_INDEXPAUSE_HD)				// Wenn ein Indexpuls innerhalb der 220ms bzw. 440ms nach dem letzen Indexpuls
			motor_auf_drehzahl = 1;					// dann luft der Motor auf Drehzahl
		else
			motor_auf_drehzahl = 0;					// sonst nicht
	} else {
		if (stopuhr < MAX_INDEXPAUSE_DD)
			motor_auf_drehzahl = 1;
		else
			motor_auf_drehzahl = 0;		
	}
	stopuhr = 0;								// Und Stopuhr wieder zurcksetzen
}		